/**
 * Java implementation of DKIM/DomainKeys. 
 * Copyright (c) 2008 Mark Boddington (www.badpenguin.co.uk)
 * 
 * This program is licensed under the terms of the GNU GPL version 2.0.
 * The DKIM specification is documented in RFC 4871
 * See: http://www.ietf.org/rfc/rfc4871.txt
 */
package badpenguin.dkim;


import java.io.IOException;
import java.io.InputStream;
import java.security.InvalidKeyException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.Signature;
import java.security.SignatureException;

import javax.naming.NamingException;
import sun.misc.BASE64Decoder;
import sun.misc.BASE64Encoder;


/**
 * This class provides all the necessary interfaces to check and verify a message
 * for either a DKIM or DomainKey signature.
 * 
 * @author Mark Boddington &lt;dk_NO_im@_SP_bad_AM_penguin.co.uk&gt;
 *         <br>http://www.badpenguin.co.uk
 */
public class Verifier {
	
	
	private Signature sig = null;
	private String sigPref = null;
	private DkimSignature dkimSig = null;
	private String mailBody = null;
	private String mailHeaders = null;
	private NSKeyStore keyStore = null;
	private boolean tryBoth = false;
	
	
	/**
	 * Create a new Verifier object. You must supply a NSKeyStore object, a
	 * signature preference, and the chose whether to try both signatures if the
	 * preferred signature is missing. The NSKeyStore will be used to lookup the 
	 * public keys for verifying the Signatures, and if a mail should contain both 
	 * key types, we will use the one you preferred. If we don't find your
	 * preferred signature and you have set arg2 to true then we will try the
	 * other type. 
	 * 
	 * @param arg0 - NSKeyStore
	 * @param arg1 - SigPref (DKIM | DomainKey)
	 * @param arg2 - Try both signature types
	 */
	public Verifier(NSKeyStore arg0, String arg1, boolean arg2){
		_Verifier(arg0,arg1,tryBoth);
		 
	}
	/**
	 * Create a new Verifier object. You must supply a NSKeyStore object, and a
	 * signature preference. Either DKIM or DomainKey. The NSKeyStore will be used
	 * to lookup the public keys for verifying the Signatures, and if a mail should
	 * contain both key types, we will use the one you preferred.
	 * <br><br>
	 * This instance will default to failing over to the other key type, should the
	 * preferred key type be missing. You can change this behaviour by calling
	 * Verifier.tryBoth(false);
	 * 
	 * @param arg0 - NSKeyStore
	 * @param arg1 - Signature Preference (DKIM | DomainKey)
	 */
	public Verifier(NSKeyStore arg0, String arg1){
		_Verifier(arg0,arg1,true);
	}
	
	/**
	 * Create a new Verifier object. You must supply a NSKeyStore object. This
	 * verifier will default to using the DKIM-Signature, but will try to use a
	 * DomainKey-Signature if a DKIM one is not present.
	 * <br><br>
	 * If you wish to stop this verifier failing back to Domainkey signatures, then
	 * you should call Verifier.tryBoth(false);
	 * 
	 * @param arg0 - NSKeyStore
	 */
	public Verifier(NSKeyStore arg0)  {
		_Verifier(arg0,"DKIM",true);
	}
	
	/**
	 * Private function called by the Verifier constructors.
	 * 
	 * @param arg0 - NSKeyStore
	 * @param arg1 - Signature Preference
	 * @param arg2 - Try both signature types
	 */
	private void _Verifier(NSKeyStore arg0, String arg1, boolean arg2) {
		tryBoth = arg2;
		sigPref = arg1;
		keyStore = arg0;
	}
	
	
	/**
	 * Set the fail back option on or off. If the signature of the type preferred
	 * is not available in the message, should we attempt to use the other type?
	 * 
	 * @param arg0 - Try other signatures?
	 */
	public void tryBoth(boolean arg0) {
		tryBoth = arg0;
	} 
	
	/**
	 * Private function to check the body data against the message digest. if the
	 * digest matches the one found in the signature, then we will return true
	 * 
	 * @return signature and body digest match
	 */
	private boolean checkBodyHash() {
		MessageDigest md = null;
		
		try {
			if ( dkimSig.getJavaAlg().equals("SHA256withRSA") ) {
				md = MessageDigest.getInstance("SHA-256");
			} else {
				md = MessageDigest.getInstance("SHA-1");
			}
		} catch (NoSuchAlgorithmException e) {
			e.printStackTrace();
		}
		
		md.update(mailBody.getBytes());
		
		BASE64Encoder bsenc = new BASE64Encoder();
		String digest = bsenc.encode(md.digest());
		
		if ( digest.equals(dkimSig.getBodyHash())) {
			return true;
		} else {
			return false;
		}
	}
	
	/**
	 * private function to check if the granularity for the key, if it exists
	 * matches the iTag, if it exists.
	 * 
	 * @param dkimSig - A DkimSignature object
	 * @param nsKey - A NSKey object
	 * @return true if g matches i, or false.
	 */
	private boolean processGranularity(DkimSignature dkimSig, NSKey nsKey) {
		String granularity = nsKey.getGranularity();
		if ( granularity.equals("*"))
			return true;
		String iTag = dkimSig.getItag();
		if ( ! iTag.equals("\0")) {
			int local = iTag.indexOf("@");
			if ( local > 0)
				if ( ! iTag.substring(0, local).matches(granularity))
					return false;
		}
		return true;
	}
	
	/**
	 * Verify the provided email message. We will scan the message for DKIM and/or
	 * DomainKey signatures and attempt to verify them. If we verify them, then we
	 * will return true.
	 * 
	 * @param msg - The message to be verified
	 * @return verified or failed
	 * @throws IOException
	 * @throws Exception
	 */
	public boolean verifyMail(InputStream msg) throws IOException, DkimException {
		
		Canonicaliser canon = new Canonicaliser(sigPref);
		MailMessage mail = new MailMessage();
		mail.processMail(msg);
		
		dkimSig = new DkimSignature(canon.initVerify(mail.getHeaders(),tryBoth));
		mailHeaders = canon.processHeaders(dkimSig);
		dkimSig.checkValidity();
		
		boolean dkim = dkimSig.isDKIM();
		
		
		mailBody = canon.processBody(mail.getBody(), dkimSig.getLtag(), dkimSig.getBodyMethod());
		
		
		NSKey nsKey = keyStore.retrieveKey(dkimSig.getDnsRecord());
		if ( dkim && ! checkBodyHash() ) {
			return false;
		}
		
		if ( ! processGranularity(dkimSig, nsKey) ) {
			if ( dkim )
				throw new DkimException(DkimError.PERMFAIL,"Inapplicable key");
			else
				throw new DkimException(DkimError.NOKEY,"Inapplicable key");
		}
		
		BASE64Decoder bs = new BASE64Decoder();
		byte[] sigBuf = bs.decodeBuffer(dkimSig.getMessageSignature());
		
		try {
			sig = Signature.getInstance(dkimSig.getJavaAlg());
			sig.initVerify(nsKey.getKey());
			sig.update(mailHeaders.getBytes());
			
			if ( ! dkim ) {
				sig.update("\r\n".getBytes());
				sig.update(mailBody.getBytes());
			}
			if ( sig.verify(sigBuf) ) {
				return true;
			} else {
				return false;
			}
		} catch (NoSuchAlgorithmException n) {
			n.printStackTrace();
		} catch (InvalidKeyException k) {
			if ( dkim )
				throw new DkimException(DkimError.PERMFAIL, "The Key found was invalid",k);
			else
				throw new DkimException(DkimError.BADFORMAT, "The Key found was invalid",k);
		} catch (SignatureException s) {
			if (dkim)
				throw new DkimException(DkimError.PERMFAIL, "Could not process the signature data",s);
			else
				throw new DkimException(DkimError.BADFORMAT, "The Key found was invalid",s);
		}
		return false;
	}
		
}


