/**
 * Java implementation of DKIM/DomainKeys. 
 * Copyright (c) 2008 Mark Boddington (www.badpenguin.co.uk)
 * 
 * This program is licensed under the terms of the GNu GPL version 2.0.
 * The DKIM specification is documented in RFC 4871
 * See: http://www.ietf.org/rfc/rfc4871.txt
 */
package badpenguin.dkim;

import java.io.InputStream;
import java.security.InvalidKeyException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.Signature;
import java.security.SignatureException;
import sun.misc.BASE64Encoder;



/**
 * The Signer class implements the necessary methods to create a DKIM or DomainKey
 * signature header for a given message. It tries to be compliant with version 1 of
 * the DKIM specification. See: http://www.ietf.org/rfc/rfc4871.txt
 * 
 * @author Mark Boddington &lt;dk_NO_im@_SP_bad_AM_penguin.co.uk&gt;
 *         <br>http://www.badpenguin.co.uk;
 */
public class Signer {
	
	/**
	 * The DKIM Signature Object to use
	 */
	DkimSignature dkimSig = null;
	
	/**
	 * A Java Security Private Key object, used for signing our messages.
	 */
	private PrivateKey privateKey = null;
	
	/**
	 * The java Signature object used to sign the message
	 */
	private Signature sig = null;
	
	/**
	 * Creates a Signer object using the specified Domain, Selector, Algorithm and 
	 * PrivateKey. This constructor also reads in the colon separated list of
	 * headers which you wish to be signed. 
	 * We will use defaults for all other DKIM options, unless you call
	 * one of the methods to modify a setting. 
	 * 
	 * @param selector - The selector for this DKIM header
	 * @param domain - The domain name for this DKIM header
	 * @param headers - A colon separated list of headers to sign
	 * @param alg - The hashing Algorithm to be used, rsa-sha256 or rsa-sha1
	 * @param key - The PrivateKey (which matches the public one found in DNS).
	 * @throws NoSuchAlgorithmException 
	 * @throws InvalidKeyException
	 */
	public Signer(String selector, String domain, String headers, String alg, PrivateKey key) throws NoSuchAlgorithmException, InvalidKeyException, DkimException {	
		dkimSig = new DkimSignature(selector, domain, headers);
		dkimSig.setAtag(alg);
		privateKey = key;
		sig = Signature.getInstance(alg);
		sig.initSign(privateKey);	
	}
	
	/**
	 * Creates a Signer object using the specified Domain, Selector and 
	 * PrivateKey. We will use defaults for all other DKIM options, unless you call
	 * one of the methods to modify their settings. 
	 * 
	 * @param selector - The selector for this DKIM header
	 * @param domain - The domain name for this DKIM header
	 * @param alg - The hashing Algorithm to be used, rsa-sha256 or rsa-sha1
	 * @param key - The PrivateKey (which matches the public one found in DNS).
	 * @throws NoSuchAlgorithmException 
	 * @throws InvalidKeyException
	 */
	public Signer(String selector, String domain, String alg, PrivateKey key) throws NoSuchAlgorithmException, InvalidKeyException, DkimException  {
		dkimSig = new DkimSignature(selector, domain);
		dkimSig.setAtag(alg);
		privateKey = key;
		sig = Signature.getInstance(dkimSig.getJavaAlg());
		sig.initSign(privateKey);	
	}
	
	
	/**
	 * Create a Signer object using the specified DkimSignature object, and the 
	 * PrivateKey. This version of the constructor allows you to control all of the
	 * settings of the DKIM signature..... Because you create it yourself ;-)
	 * 
	 * @param DKIMSig
	 * @param key
	 * @throws NoSuchAlgorithmException
	 * @throws InvalidKeyException
	 * @throws DkimException
	 */
	public Signer(DkimSignature DKIMSig, PrivateKey key) throws NoSuchAlgorithmException, InvalidKeyException, DkimException {
		dkimSig = DKIMSig;
		privateKey = key;	
		sig = Signature.getInstance(dkimSig.getJavaAlg());
		sig.initSign(privateKey);		
	}
	
	/**
	 * This private function is used by the signMail method to generate the body
	 * hash. It expects to be given the mail body, and returns a BASE64 encoded
	 * digest of the data.
	 * 
	 * @param mailBody - The body data of the mail message
	 * @return BASE64 encoded digest
	 */
	private String genBodyHash(String mailBody) {
		
		MessageDigest md = null;
		
		try {
			if ( dkimSig.getJavaAlg().equals("SHA256withRSA") ) {
				md = MessageDigest.getInstance("SHA-256");
			} else {
				md = MessageDigest.getInstance("SHA-1");
			}
		} catch (NoSuchAlgorithmException e) {
			e.printStackTrace();
		}
		
		md.update(mailBody.getBytes());
		
		BASE64Encoder bsenc = new BASE64Encoder();
		String digest = bsenc.encode(md.digest());
		
		return digest;
	}
	
	/**
	 * Create a DKIM-Signature header for the provided mail message. This function
	 * expects to receive a BufferedInputStream containing the raw email message. 
	 * That means both the headers and the body as transfered in SMTP data, 
	 * excluding the ending &lt;CRLF&gt;.&lt;CRLF&gt;
	 * 
	 * @param msg - The raw email message (headers + body)
	 * @return DKIM-Signature
	 */
	public String signMail(InputStream msg) throws DkimException {

		Canonicaliser canon = new Canonicaliser();
		MailMessage mail = new MailMessage();
		mail.processMail(msg);
		
		canon.initSign(dkimSig.getDkimSig(), mail.getHeaders());
		String mailHeaders = canon.processHeaders(dkimSig);
		String mailBody = canon.processBody(mail.getBody(), dkimSig.getLtag(), dkimSig.getBodyMethod());
		
		String digest = genBodyHash(mailBody);
		
		dkimSig.setBHtag(digest);
		
		try {
			sig.update(mailHeaders.getBytes());
			BASE64Encoder bsenc = new BASE64Encoder();
			dkimSig.setBtag(bsenc.encode(sig.sign()));
		} catch (SignatureException e) {
			e.printStackTrace();
		}
				
		return dkimSig.getDkimSig();
	}

}
