/**
 * Java implementation of DKIM/DomainKeys. 
 * Copyright (c) 2008 Mark Boddington (www.badpenguin.co.uk)
 * 
 * This program is licensed under the terms of the GNU GPL version 2.0.
 * The DKIM specification is documented in RFC 4871
 * See: http://www.ietf.org/rfc/rfc4871.txt
 */
package badpenguin.dkim;


import java.util.Hashtable;

import javax.naming.NamingException;
import javax.naming.directory.Attributes;
import javax.naming.directory.DirContext;
import javax.naming.directory.InitialDirContext;


/**
 * The NSKeyStore can be used to retrieve keys from a Naming Service (Currently
 * only DNS) and store them in a hash table. Any subsequent requests for the same 
 * domain key record are returned directly from the hash table, thus reducing the
 * number of DNS queries performed during verification.
 * 
 * @author Mark Boddington &lt;dk_NO_im@_SP_bad_AM_penguin.co.uk&gt;
 *         <br>http://www.badpenguin.co.uk
 */
public class NSKeyStore {
	
	private static Hashtable<String,NSKey> keyMap = null;
	private DirContext dnsCtx = null;

	public NSKeyStore(String type, String dnsServer) throws NamingException {
			
		if ( ! type.equals("dns"))
			throw new NamingException("Only the DNS name service is supported");
		
		keyMap = new Hashtable<String,NSKey>();
		Hashtable<String,String> env = new Hashtable<String,String>();	
		env.put("java.naming.factory.initial", "com.sun.jndi.dns.DnsContextFactory");
		env.put("java.naming.provider.url", "dns://" + dnsServer );
		dnsCtx = new InitialDirContext(env);	
	}
	
	/**
	 * This method returns a NSKey object for the give domain key record. If the
	 * lookup has been performed before, then the NSKey will be returned from the
	 * hash table, otherwise a Name Service query will be initiated and the result
	 * will be stored and returned.
	 * 
	 * @param lookup - The domain key record to retrieve
	 * @return The NSKey for this request
	 * @throws DkimException - The NameService lookup has failed.
	 */
	public NSKey retrieveKey(String lookup) throws DkimException {
		
		if ( keyMap.containsKey(lookup)) {
			return keyMap.get(lookup);
		} else {
			
			Attributes attrs = null;
			try {
				attrs = dnsCtx.getAttributes(lookup, new String[] {"txt"});
			} catch (NamingException e) {
				if ( e.getMessage().contains("name not found") ) {
					throw new DkimException(DkimError.PERMFAIL,"No key for signature found at " + lookup, e);
				}
				throw new DkimException(DkimError.TEMPFAIL, "Failed to perform NameService lookup", e);
			}
			String record = attrs.toString();
			NSKey nsKey = new NSKey(record);
			
			synchronized(keyMap) {
				if ( ! keyMap.contains(lookup))
					keyMap.put(lookup, nsKey);
			}		
			return nsKey;	
		}
	}
	
}
