/**
 * Java implementation of DKIM/DomainKeys. 
 * Copyright (c) 2008 Mark Boddington (www.badpenguin.co.uk)
 * 
 * This program is licensed under the terms of the GNU GPL version 2.0.
 * The DKIM specification is documented in RFC 4871
 * See: http://www.ietf.org/rfc/rfc4871.txt
 */
package badpenguin.dkim;

import java.security.PublicKey;
import java.util.Scanner;

import sun.misc.BASE64Decoder;
import sun.security.rsa.RSAPublicKeyImpl;


/**
 * This Object is used to store a public key and the associated meta data retrieved
 * from DNS (or maybe some other naming system in future). Typically these keys are
 * created and stored in a HashTable by the NSKeyStore object, which means we don't
 * have to make a DNS call every time the DKIM library parses a mail message.
 * 
 * @author Mark Boddington &lt;dk_NO_im@_SP_bad_AM_penguin.co.uk&gt;
 *         <br>http://www.badpenguin.co.uk
 */
public class NSKey {
	
	private PublicKey key;
	private String version = "DKIM1";
	private String granularity = "*";
	private String hash = "sha1:sha256";
	private String type = "rsa";
	private String notes = null;
	private String service = "*";
	
	// flags
	private boolean testing = false;
	private boolean noSubdomains = false;
	
	/**
	 * Create a new NSKey object, from the Name Service text record provided in 
	 * the record string. This string should be a DNS txt record which adheres
	 * to the _domainkey DNS TXT record specification
	 * 
	 * @param record - The Name Services TXT record
	 */
	public NSKey(String record) {
			
		Scanner keyScan = new Scanner(record);
		keyScan.useDelimiter("[\\;]");
		keyScan.next();
		while ( keyScan.hasNext()) {
			String line = keyScan.next().replaceAll("[\t\" ]", "");
			String tag[] = line.trim().split("=",2);
			if ( tag[0].equals("g")) {
				granularity=tag[1];
			} else if (tag[0].equals("t")) {
				int i=0;
				if ( tag[1].contains("y") ) {
					testing = true;
					i++;
				} else if ( tag[1].contains("s")) {
					noSubdomains = true;
				} 
			} else if ( tag[0].equals("p")) {
				try{	
					BASE64Decoder decoder = new BASE64Decoder();
					byte[] decoded = decoder.decodeBuffer(tag[1]);				
					key = new RSAPublicKeyImpl( decoded );
					
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}	
	}
	
	/**
	 * Retrieve the public key from thise NSKey object. The Public key can then be
	 * used for verification tasks.
	 * 
	 * @return The RSA Public Key
	 */
	public PublicKey getKey() {
		return key;
	}
	
	/**
	 * Returns true if this NSKey record has the "y" flag set. Indicating that this
	 * domain is currently testing DKIM, Failed messages should be treated as though
	 * they had no signature.
	 * 
	 * @return True or False
	 */
	public boolean isTesting() {
		return testing;
	}
	
	/**
	 * Returns true if this NSKey record has the s flag set. The s flag indicates
	 * that if a DKIM-Signature has the i tag set, then it can not be a subdomain
	 * of the d tag. That is the right hand side of @ in both i and d tags must
	 * match exactly.
	 * 
	 * @return True or False
	 */
	public boolean noSubdomains() {
		return noSubdomains;
	}
	
	public String getGranularity() {
		return granularity;
	}

}
