/**
 * Java implementation of DKIM/DomainKeys. 
 * Copyright (c) 2008 Mark Boddington (www.badpenguin.co.uk)
 * 
 * This program is licensed under the terms of the GNU GPL version 2.0.
 * The DKIM specification is documented in RFC 4871
 * See: http://www.ietf.org/rfc/rfc4871.txt
 */
package badpenguin.dkim;

/**
 * This class enumerates the possible DKIM and DomainKey errors specified in their
 * respective RFC's. The JavaDKIM library should through a DkimException with the
 * appropriate DkimError when ever something is not quite right.
 * 
 * @author Mark Boddington &lt;dk_NO_im@_SP_bad_AM_penguin.co.uk&gt;
 *         <br>http://www.badpenguin.co.uk
 */
public enum DkimError {
	
	/**
	 * DomainKey status - The signature was verified at the time of testing
	 */
	GOOD ("good", "The signature was verified at the time of testing"),
	/**
	 * DomainKey status - The signature failed verification
	 */
	BAD ("bad", "The signature failed verification"),
	/**
	 * DomainKey status - The public key query failed as the key does not exist
	 */
	NOKEY ("no key", "The public key query failed as the key does not exist"),
	/**
	 * DomainKey status - The public key query failed as the key has been revoked
	 */
	REVOKED ("revoked", "The public key query failed as the key has been revoked"),
	/**
	 * DomainKey status - This email has no DomainKey-Signature header
	 */
	NOSIG ("no signature", "This email has no DomainKey-Signature header"),
	/**
	 * DomainKey status - The signature or the public key contains unexpected data
	 */
	BADFORMAT ("bad format", "The signature or the public key contains unexpected data"),
	/**
	 * DomainKey status - The sending domain has indicated it does not participate in DomainKeys
	 */
	PARTICIPANT ("non-participant", "The sending domain has indicated it does not participate in DomainKeys"),
	/**
	 * DKIM status - Permenant Failure encountered. Message will never validate
	 */
	PERMFAIL ("PERMFAIL", "Permenant Failure encountered. Message will never validate"),
	/**
	 * DKIM status - Temporary Failure encountered. Message may validate later
	 */
	TEMPFAIL ("TEMPFAIL", "Temporary Failure encountered. Message may validate later"),
	/**
	 * Library Error - An internal error has occurred
	 */
	LIBERROR ("Library Error", "An internal error has occurred");
	
	private String status;
	private String description;
	
	/**
	 * Construct a new enumeration of DkimError
	 * 
	 * @param arg1 - The RFC error
	 * @param arg2 - The Description of the error.
	 */
	private DkimError(String arg1, String arg2) {
		status = arg1;
		description = arg2;
	}
	
	/**
	 * Get the RFC error status
	 * @return The RFC error
	 */
	public String getStatus() {
		return status;
	}
	
	/**
	 * Get the detailed description
	 * @return The Description
	 */
	public String getDescription() {
		return description;
	}

}
