/**
 * Java implementation of DKIM/DomainKeys. 
 * Copyright (c) 2008 Mark Boddington (www.badpenguin.co.uk)
 * 
 * This program is licensed under the terms of the GNU GPL version 2.0.
 * The DKIM specification is documented in RFC 4871
 * See: http://www.ietf.org/rfc/rfc4871.txt 
 */
package badpenguin.dkim;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.Scanner;
import java.util.Stack;
import java.util.regex.Pattern;


/**
 * The Canonicaliser class is responsible for preparing a message for signing or 
 * verification. It understands the DKIM canonical methods simple and relaxed, as 
 * well as the DomainKey methods simple and nofws.
 * 
 * @author Mark Boddington &lt;dk_NO_im@_SP_bad_AM_penguin.co.uk&gt;
 *         <br>http://www.badpenguin.co.uk
 */
public class Canonicaliser {
	
	// The three canonical mathods used in this canonicaliser
	private static final CanonicalMethod SIMPLE = CanonicalMethod.SIMPLE;
	private static final CanonicalMethod RELAXED = CanonicalMethod.RELAXED;
	private static final CanonicalMethod NOFWS = CanonicalMethod.NOFWS;
	
	
	private boolean useDKIM = false;
	private boolean fallBack = false;
	private Stack<String> headerStack = null;
	private String signHeaders = null;
	
	/**
	 * Create a new canonicaliser instance, which will prefer to use the Signature
	 * type specified in sigPref (either "DomainKey" or "DKIM"). If you initialise
	 * this instance for signing, then we will create the appropriate header based
	 * on this preference. If this instance is initialised for Verification, then
	 * we will attempt to verify the header of this preference, if both are present.
	 * 
	 * @param sigPref - Singature preference, DKIM or DomainKey
	 */
	public Canonicaliser(String sigPref) {
		if ( sigPref.equalsIgnoreCase("DKIM")) {
			useDKIM = true;
		} else if ( sigPref.equalsIgnoreCase("DomainKey")) {
			useDKIM = false;
		} else {
			//invalid option, preferring DKIM
			useDKIM = true;
		}
		fallBack = false;
	}
	

	/**
	 * Create a new canonicaliser instance. This instance will prefer to use the
	 * DKIM signature. If you initialise this instance for signing, then we will 
	 * create a DKIM-Signature. If you initialise for Verification, then we will
	 * chose a DKIM-Signature over a DomainKey-Signature, should both be present.
	 */
	public Canonicaliser() {
		useDKIM = true;
		fallBack = false;
	}

	/**
	 * Take a line of input and canonicalise it in accordance with the method
	 * specified. Returns the canonicalised version of this line.
	 * @param line - The line to canonicalise
	 * @param method - The method to use (simple|relaxed|nofws)
	 * @return The canonicalised input
	 */
	protected String processLine(String line, CanonicalMethod method) {
		
		if ( method.equals(NOFWS) ) {
			line = line.replaceAll("[\t\r\n ]", "");
			line += "\r\n";
		} else if ( method.equals(RELAXED) ) {
			line = line.replaceAll("[\r\n\t ]+", " ");
			line = line.replaceAll("(?m)[\t\r\n ]+$", "");
			line += "\r\n";
		} else if (method.equals(SIMPLE) ) {
			line += "\r\n";
		} 
		return line;
	}
	
	/**
	 * Read the body from the given byte stream and process it with the specified
	 * canonicalisation method. If the length argument is greater than -1, then we
	 * will truncate the body data to the given length. this method returns the
	 * body data in a String, ready to be signed or verified.
	 * 
	 * @param bodyStream - The body data to be processed
	 * @param length - The length at which to truncate the body, or -1
	 * @param method - The CanonicalMethod to use
	 * @return A string containing the canonicalised body
	 */
	public String processBody(ByteArrayOutputStream bodyStream, long length, 
			CanonicalMethod method) {

		String mailBody = "";
		
		// if the DKIM L tag was 0, then return just CRLF
		if ( length == 0 ) {
			return "\r\n";
		}
		
		Scanner mail =  new Scanner(bodyStream.toString());
		mail.useDelimiter(Pattern.compile("[\r\n]"));
		
		while (mail.hasNextLine()) {
			String line = mail.nextLine();
			mailBody += processLine(line,method);
		}
		
		mailBody = mailBody.replaceAll("[\t \r\n]+$", "\r\n");
		return mailBody; //+ "\r\n";
	}
	

	/**
	 * Read the headers from the given byte stream and process them with the 
	 * specified canonicalisation method. If the length argument is greater than 
	 * -1, then we will truncate the body data to the given length. this method 
	 * returns the headers in a String, ready to be signed or verified.
	 * @param dkimSig - The DkimSignature object which relates to these headers
	 * @return The processed headers, ready for signing/verifying
	 * @throws DkimException
	 */
	public String processHeaders(DkimSignature dkimSig) throws DkimException {
	
		// We have already run processHeaders since init. The stack will be empty
		// return the signHeaders we already have.
		if ( signHeaders != null ) {
			return signHeaders;
		}
		
		// Clear the signHeaders
		signHeaders = "";

		// If the stack is empty, then we haven't been initialised. Throw an error
		if (headerStack == null ) {
			throw new DkimException(DkimError.LIBERROR,"You must call initHeaders first!");
		}
		
		// find out what's what from the DkimSignature object.
		String sigHtag = dkimSig.getHtag();
		CanonicalMethod method = dkimSig.getHeaderMethod();
		boolean isDKIM = dkimSig.isDKIM();
		
		// If we are processing DKIM and the H tag is empty, throw an error.
		if ( isDKIM ) {		
			if ( sigHtag == null || sigHtag.isEmpty() ) {
				throw new DkimException(DkimError.PERMFAIL, "The madatory H tag appears to be missing from the DKIM-Signature");
			}	
		} 
		
		if ( sigHtag != "\0") {
			
			// If we're DKIM, add the dkim-signature to the end of the headers.
			if ( isDKIM ) {
				sigHtag += ":dkim-signature";
			}
			
			String[] headers = sigHtag.split(":");
			int headerCount = headers.length;
			String[] values = new String[headerCount];
			
			while ( ! headerStack.isEmpty() ) {
				String header = headerStack.pop();
				for ( int index = 0 ; index < headerCount ; index++ ) {
					if ( header.equalsIgnoreCase(headers[index]) ) {
						headers[index] = header;
						if ( values[index] == null ) {
							values[index] =  processLine(headerStack.pop(),method).trim();
							if ( headers[index].equalsIgnoreCase("dkim-signature"))
								values[index] = values[index].replaceAll("b=[A-Za-z0-9+/= ]+", "b=");
							break;
						}
					}
				}
			}
			
			for (int index = 0 ; index < headers.length ; index++ ) {
				if (method.equals(NOFWS)) {
					signHeaders += processLine(headers[index] + ":" + values[index],method);
				} else if ( method.equals(RELAXED) ){
					signHeaders += processLine(headers[index].toLowerCase() + ":" + values[index],method);
				}
			}
				
		} else {		
			// We must be a DomainKey, with no headers, this will never work!
			// in fact, £50, this will fail! :-p
			while ( ! headerStack.isEmpty() ) {
				String header = headerStack.pop();
				String value = headerStack.pop();
				if ( header.equalsIgnoreCase("DomainKey-Signature"))
					continue;
				signHeaders = processLine(header + ":" + value, method) + signHeaders;
			}	
		}
			
		if ( isDKIM )
			signHeaders = signHeaders.replaceAll("(?m)[\r\n]+$", "");
		return signHeaders;
	}
				
	/**
	 * Initialise this Canonicaliser for Verification. We will read in the headers
	 * from the message, and they will be placed onto a stack for later processing
	 * with processHeaders(). This method will return either the DKIM or DomainKey
	 * signature based on the preference set during initialisation. If the fallback
	 * option is set to true, then we will return the other signature, should the
	 * preferred signature be unavailable.
	 * 
	 * @param headerStream - The message headers to be read
	 * @param fallback - Should we fall back to the other DomainKey header?
	 * @return The DKIM-Signature or DomainKey-Signature
	 * @throws DkimException
	 */
	public String initVerify(ByteArrayOutputStream headerStream, boolean fallback) throws DkimException{
		
		headerStack = new Stack<String>();
		signHeaders = null;
		fallBack = fallback;

		return init(headerStream);
	}
	
	/**
	 * Initialise this canonicaliser for use in signing. We require a DKIM-signature,
	 * and the message headers. We will push the headers and the signature onto a 
	 * stack for later processing with processHeaders().
	 * 
	 * @param DKIMSig - The DKIM or DomainKey signature.
	 * @param headerStream - The message headers to be read.
	 * @throws DkimException
	 */
	public void initSign(String DKIMSig, ByteArrayOutputStream headerStream) throws DkimException {

		headerStack = new Stack<String>();
		signHeaders = null;
		
		try {
			headerStream.write(DKIMSig.getBytes());
		} catch (IOException e) {
			e.printStackTrace();
		}
		init(headerStream);
	}
	
	/**
	 * This function performs most of the work for the public initVerify() and
	 * initSign() methods. It takes in the message headers and pushes them onto a
	 * stack. We return a DKIM or DomainKey signature based on the signature
	 * preference and the fall back option.
	 * 
	 * @param headerStream - The messages headers.
	 * @return The DKIM or DomainKey header
	 * @throws DkimException
	 */
	private String init(ByteArrayOutputStream headerStream) throws DkimException {
		
		String domKeySig = null;
		String dkimSig = null;
		
		Scanner mail =  new Scanner(headerStream.toString());
		mail.useDelimiter(Pattern.compile("[\r\n]"));
		
		while (mail.hasNextLine()) {
			
			String line = mail.nextLine();
			while ( mail.hasNextLine()) {
				
				if ( mail.hasNext(Pattern.compile("(?m)^\\s+.*")) ) {
					line += "\r\n" + mail.nextLine();
				} else {
					break;
				}
			}
			
			int colon = line.indexOf(':');
			headerStack.push(line.substring(colon+1) );
			headerStack.push(line.substring(0, colon) );
						
			if ( line.startsWith("DomainKey-Signature")) {
				line = line.replaceAll("[\t\r\n ]+", "");
				domKeySig = line.trim();
			} else if (line.startsWith("DKIM-Signature") ) {
				line = line.replaceAll("[\r\n\t ]+", "");
				//line = line.replaceAll("[\t ]+", " ");
				dkimSig = line.trim();
			}	
		}
		
		if ( useDKIM && dkimSig != null) {
			return dkimSig;
		} else if ( !useDKIM && domKeySig != null) {
			return domKeySig;
		} else if ( fallBack && useDKIM && domKeySig != null ) {
			return domKeySig;
		} else if ( fallBack && !useDKIM && dkimSig != null ) {
			return dkimSig;
		} else if ( useDKIM ) {
			throw new DkimException(DkimError.PERMFAIL, "No DKIM-Signature header was found");
		} else {
			throw new DkimException(DkimError.NOSIG);
		}
		
	}
	
}
